/* -*-c++-*- Copyright (C) 2018 Advanced Driver Information Technology.
 *
 * This library is open source and may be redistributed and/or modified under
 * the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
 * (at your option) any later version.  The full license is in LICENSE file
 * included with this distribution, and on the openscenegraph.org website.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * OpenSceneGraph Public License for more details.
*/


#include <errno.h>
#include <string>
#include "WLIVIShell.hpp"
#include "WLSurfaceEvent.hpp"

using namespace WaylandBackend;

const struct ivi_surface_listener WLIVIShell::surface_listener =
{
    WLIVIShell::handleSurfaceConfigure
};


WLIVIShell::WLIVIShell(struct wl_event_queue *eventQ)
:_wlEventQ(eventQ)
,_wlID(0)
,_wlInterfaceName("ivi_application")
,_iviApplication(NULL)
,_shellName("IVIShell")
{

}

int WLIVIShell::init(struct wl_registry* registry, uint32_t name,
             const char* interface, uint32_t version)
{
    int ret = -EINVAL;
    std::string ifName(interface);

    if (ifName == _wlInterfaceName)
    {
        _wlID = name;

        ret = 0;

        if (NULL != _wlEventQ)
        {
            struct wl_registry* wrappedRegistry = (struct wl_registry*)
                             wl_proxy_create_wrapper((void *)registry);
            if (NULL != wrappedRegistry)
            {
                wl_proxy_set_queue((struct wl_proxy*)wrappedRegistry,
                                   _wlEventQ);
                _iviApplication = (struct ivi_application*)
                                   wl_registry_bind(registry, name,
                                                    &ivi_application_interface,
                                                    version);
                wl_proxy_wrapper_destroy(wrappedRegistry);
            }
        }
        else
        {
            /*
            * It is not good idea to use default queue. default queue can be
            * dispatched by undesired threads
            */
            _iviApplication = (struct ivi_application*)
                              wl_registry_bind(registry, name,
                                               &ivi_application_interface,
                                               version);
        }
        if (NULL == _iviApplication)
        {
            ret = -ENOMEM;
        }
    }
    return ret;
}

void
WLIVIShell::handleSurfaceConfigure(void *data,
                                   struct ivi_surface *ivi_surface,
                                   int32_t width,
                                   int32_t height)
{
    WLSurfaceEvent *surfEvent;

    surfEvent = static_cast<WLSurfaceEvent *>
                     (ivi_surface_get_user_data(ivi_surface));
    if (NULL != surfEvent)
        surfEvent->surfaceConfigured(width, height);
}


int
WLIVIShell::surfaceCreate(struct wl_surface* wlSurface,
                          unsigned int surfaceId,
                          void **shellSurface)
{
    int ret = -EINVAL;

    if (NULL != shellSurface)
    {
        *shellSurface = (void *)ivi_application_surface_create
                                (_iviApplication, surfaceId, wlSurface);
        if (NULL != *shellSurface)
        {
            ivi_surface_add_listener((struct ivi_surface*) *shellSurface,
                    &surface_listener, this);

            ivi_surface_set_user_data((struct ivi_surface*) *shellSurface,
                    wl_surface_get_user_data(wlSurface));

            ret = 0;
        }
    }
    return ret;
}


int
WLIVIShell::surfaceDestroy(void* shellSurface)
{
    int ret = -EINVAL;
    if (NULL != shellSurface)
    {
        ivi_surface_destroy((struct ivi_surface*)shellSurface);
        ret = 0;
    }
    return ret;
}

uint32_t
WLIVIShell::getWLID()
{
    return _wlID;
}

std::string&
WLIVIShell::getWLName()
{
    return _wlInterfaceName;
}

WLIVIShell::~WLIVIShell()
{
    if (NULL != _iviApplication)
    {
        ivi_application_destroy (_iviApplication);
    }
}
